% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [theta] = InitCond(theta_L,theta_R,z,ampl)
% InitCond  compute the initial condition of the director field.
%
% [theta] = InitCond(theta_L,theta_R,nz)
% with theta_L and theta_U scalars.  Returns an array of dimensions
% (nz,1) that interpolates between the scalars theta_L and theta_R in the
% z-coordinate.    The values of the z coordinate are the points of a
% Gauss-Lobatto grid.
%
%  [a] = InitCond(theta_L,theta_U,nz,state,ampl) 
% has the same effect as the previous command, but adds a uniformly
% distributed random noise of amplitude "ampl" to the field at the interior
% points of the Gauss-Lobatto grid.   "state" is an integer scalar used to
% set the state of the random number generator.

theta = theta_L + (theta_R-theta_L)*(z-min(z))/(max(z)-min(z));

% Optionally add a small sinusoidal perturbation
if (nargin > 3)
  theta = theta + ampl*sin(pi*(z-min(z))/(max(z)-min(z)));
end
end